#include "fdm.h"
#include <stdlib.h>

FDM::FDM()
{
  ix = 3; //43;
  jy = 3; //33;
  IJ = ix*jy;
  dx = 1.;
  dy = 1.;
  dt = 0.25; // sec
  S0 = 1e-5;
  Kf = 1e-5;                                     // m/s
  //Q = 3.171e-9; // 100mm/year : 0.1/365/86400 = 3.171e-9 m/s
  Q = 0.; //1e-5;
  u0 = 0.;
  x0 = 0.; //4423656.0991422;
  y0 = 0.; //5716944.1927754;
  //memory allocation
  u.resize(IJ);
  u_new.resize(IJ);
  //output
  out_file.open("out.txt");
  eqs_file.open("eqs.txt");
  dx2 = dx*dx;
  dy2 = dy*dy;
  Ne = (Kf/S0) * (dt/dx2);
  matrix = new double[IJ*IJ];
  vecb = new double[IJ];
  vecx = new double[IJ];
}

void FDM::SetInitialConditions()
{
  for(j=0;j<jy;j++)
  {
    nn = j*ix;
    for( i=0;i<ix;i++)
    {
      n = nn+i;
      u[n] = u0;
      u_new[n] = u0;
    }
  }
}

void FDM::SetBoundaryConditions()
{
  bc_nodes.push_back(0); u[0] = 1;  u_new[0] = 1;
  bc_nodes.push_back(8); u[IJ-1] = -1;  u_new[IJ-1] = -1;
}

void FDM::RunTimeStep()
{
}

void FDM::SaveTimeStep()
{
  //save time step
  for(int j=0;j<jy;j++)
    for(int i=0;i<ix;i++)
    {
      u_new[j*ix+i] = vecx[j*ix+i];
      u[j*ix+i] = u_new[j*ix+i];
    }
}

void FDM::OutputResults(int t)
{
  if((t%1)==0)
  {
    out_file << "ZONE T=\"BIG ZONE\", I=" << ix << ", J=" << jy <<", DATAPACKING=POINT"  << std::endl;
    for(int j=0;j<jy;j++) //y
    {
      y = y0 + j*dy;
      nn = j*ix;
      for(int i=0;i<ix;i++) //x
      {
        n = nn+i;
        x = x0 + i*dx;
        out_file << x << "\t" << y << "\t" << u_new[n] << std::endl;
      }
    }
  }
}

void FDM::OutputMesh()
{
  //-----------------------------------------------------------------------
  //output:MSH
  std::ofstream msh_file;
  msh_file.open("2dfd.msh");
  msh_file << "#FEM_MSH" << std::endl;
  msh_file << " $PCS_TYPE" << std::endl;
  msh_file << "  GROUNDWATER_FLOW" << std::endl;
  msh_file << " $NODES" << std::endl;
  msh_file << "  " << ix*jy << std::endl;
  for(int j=0;j<jy;j++)
  {
    y = y0 + j*dy;
    nn = j*ix;
    for(int i=0;i<ix;i++)
    {
      n = nn+i;
      x = x0 + i*dx;
      msh_file << n << "\t" << x << "\t" << y << "\t" << 0 << std::endl;
    }
  }
  msh_file << " $ELEMENTS" << std::endl;
  msh_file << "  " << (ix-1)*(jy-1) << std::endl;
  int c=0;
  for(int j=0;j<jy-1;j++)
  {
    nn = j*ix;
    for(int i=0;i<ix-1;i++)
    {
      n = nn+i;
      msh_file << c++ << "\t" << 0 << " quad " << n << "\t" << n+1 << "\t" << n+ix+1 << "\t" << n+ix << std::endl;
    }
  }
  msh_file << "#STOP" << std::endl;
}

void FDM::AssembleEquationSystem()
{
  // Matrix entries
  for(i=0;i<IJ;i++)
  {
    vecb[i] = u[i];
    for(j=0;j<IJ;j++)
    {
      matrix[i*IJ+j] = 0.0;
      if(i==j)
        matrix[i*IJ+j] = 1. + 4.*Ne;
      else if(abs((i-j))==1)
        matrix[i*IJ+j] = - Ne;
    }
  }
  // Incorporate boundary conditions
  IncorporateBoundaryConditions();
  // Matrix output
  WriteEquationSystem();
}

void FDM::WriteEquationSystem()
{
  eqs_file << "--->" << std::endl;
  for(i=0;i<IJ;i++)
  {
    for(j=0;j<IJ;j++)
    {
      eqs_file << matrix[i*IJ+j] << "\t";
    }
    eqs_file << "b:" << vecb[i] << " ";
    eqs_file << std::endl;
  }
}

void FDM::IncorporateBoundaryConditions()
{
  //------------------------------------
  size_t i_bc;
  int i_row, k;
  for(i_bc=0;i_bc<bc_nodes.size();i_bc++)
  {
     i_row = bc_nodes[i_bc];
     // Null off-diangonal entries of the related row and columns
     // Apply contribution to RHS by BC
     for(j=0;j<IJ;j++)
     {
        if(i_row == j)          
           continue; // do not touch diagonals
        matrix[i_row*(IJ)+j] = 0.0; // NUll row
        k = j*(IJ)+i_row;
        // Apply contribution to RHS by BC
        vecb[j] -= matrix[k]*u[i_row];
        matrix[k] = 0.0; // Null column
     }
     // Apply Dirichlet BC
     vecb[i_row] = u[i_row]*matrix[i_row*(IJ)+i_row];
  } 
}
